from tianshou.env import BaseVectorEnv, SubprocVectorEnv
from typing import Any, List, Optional, Tuple, Union
import numpy as np

class GCVectorEnv(SubprocVectorEnv):
    """Vectorized environment wrapper based on subprocess for GC RL.

    .. seealso::

        Please refer to :class:`~tianshou.env.BaseVectorEnv` for other APIs' usage.
    """

    def reset(
        self,
        id: Optional[Union[int, List[int], np.ndarray]] = None,
        kwarg_list = None,
        **kwargs: Any,
    ) -> Tuple[np.ndarray, Union[dict, List[dict]]]:
        """Reset the state of some envs and return initial observations.

        If id is None, reset the state of all the environments and return
        initial observations, otherwise reset the specific environments with
        the given id, either an int or a list.
        """
        self._assert_is_not_closed()
        id = self._wrap_id(id)
        if self.is_async:
            self._assert_id(id)

        # send(None) == reset() in worker
        for i in id:
            if kwarg_list is not None:
                self.workers[i].send(None, **kwarg_list[i])
            else:
                self.workers[i].send(None, **kwargs)
        ret_list = [self.workers[i].recv() for i in id]

        assert (
            isinstance(ret_list[0], (tuple, list)) and len(ret_list[0]) == 2
            and isinstance(ret_list[0][1], dict)
        ), "The environment does not adhere to the Gymnasium's API."

        obs_list = [r[0] for r in ret_list]

        if isinstance(obs_list[0], tuple):  # type: ignore
            raise TypeError(
                "Tuple observation space is not supported. ",
                "Please change it to array or dict space",
            )
        try:
            obs = np.stack(obs_list)
        except ValueError:  # different len(obs)
            obs = np.array(obs_list, dtype=object)

        infos = [r[1] for r in ret_list]
        return obs, infos  # type: ignore